import SwiftUI
import UserNotifications

struct DisclaimerView: View {
    @State private var hasAgreed = false
    @State private var showingDetailedDisclaimer = false
    @Binding var hasAcceptedDisclaimer: Bool
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            ScrollView {
                VStack(spacing: 20) {
                    // Header
                    VStack(spacing: 12) {
                        Image(systemName: "exclamationmark.triangle.fill")
                            .font(.system(size: 40))
                            .foregroundColor(.orange)
                        
                        Text("Important Medical Disclaimer")
                            .font(.title2)
                            .fontWeight(.bold)
                            .multilineTextAlignment(.center)
                    }
                    .padding(.top)
                    
                    // Main disclaimer content
                    VStack(alignment: .leading, spacing: 16) {
                        DisclaimerSection(
                            title: "Entertainment & Educational Purpose Only",
                            content: "This app is designed for entertainment and educational purposes only. It is NOT a medical device, diagnostic tool, or health assessment instrument."
                        )
                        
                        DisclaimerSection(
                            title: "Not Medical Advice",
                            content: "The calculations, predictions, and information provided by this app do not constitute medical advice, diagnosis, or treatment recommendations."
                        )
                        
                        DisclaimerSection(
                            title: "Consult Healthcare Professionals",
                            content: "Always consult with qualified healthcare professionals for medical advice, diagnosis, and treatment decisions."
                        )
                        
                        DisclaimerSection(
                            title: "Hypothetical Calculations",
                            content: "All predictions are based on statistical models and general population data. Individual results may vary significantly."
                        )
                    }
                    .padding(.horizontal)
                    
                    // Agreement toggle
                    VStack(spacing: 12) {
                        Toggle(isOn: $hasAgreed) {
                            Text("I understand and agree to these terms")
                                .font(.footnote)
                                .fontWeight(.medium)
                        }
                        .toggleStyle(SwitchToggleStyle(tint: .blue))
                        
                        Button("View Full Disclaimer") {
                            showingDetailedDisclaimer = true
                        }
                        .font(.caption)
                        .foregroundColor(.blue)
                    }
                    .padding(.horizontal)
                    
                    // Action buttons
                    VStack(spacing: 12) {
                        Button("Accept and Continue") {
                            acceptDisclaimer()
                        }
                        .buttonStyle(.borderedProminent)
                        .disabled(!hasAgreed)
                        
                        Button("Decline") {
                            declineDisclaimer()
                        }
                        .buttonStyle(.bordered)
                        .foregroundColor(.secondary)
                    }
                    .padding(.horizontal)
                    .padding(.bottom)
                }
            }
            .navigationTitle("Disclaimer")
            .navigationBarTitleDisplayMode(.inline)
            .sheet(isPresented: $showingDetailedDisclaimer) {
                DetailedDisclaimerView()
            }
        }
    }
    
    private func acceptDisclaimer() {
        // Store user consent
        UserDefaults.standard.set(true, forKey: "hasAcceptedMedicalDisclaimer")
        UserDefaults.standard.set(Date(), forKey: "disclaimerAcceptanceDate")
        
        // Log consent for compliance
        logDisclaimerAcceptance()
        
        hasAcceptedDisclaimer = true
        dismiss()
    }
    
    private func declineDisclaimer() {
        // User declined - close app or navigate to info screen
        hasAcceptedDisclaimer = false
        dismiss()
    }
    
    private func logDisclaimerAcceptance() {
        // Log acceptance for audit trail (privacy compliant)
        let acceptanceData: [String: Any] = [
            "event": "disclaimer_accepted",
            "timestamp": Date().timeIntervalSince1970,
            "version": "1.0"
        ]
        
        // Store locally for compliance verification
        UserDefaults.standard.set(acceptanceData, forKey: "disclaimerAcceptanceLog")
    }
}

struct DisclaimerSection: View {
    let title: String
    let content: String
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            Text(title)
                .font(.headline)
                .fontWeight(.semibold)
                .foregroundColor(.primary)
            
            Text(content)
                .font(.body)
                .foregroundColor(.secondary)
                .fixedSize(horizontal: false, vertical: true)
        }
        .frame(maxWidth: .infinity, alignment: .leading)
        .padding()
        .background(Color.gray.opacity(0.1))
        .cornerRadius(12)
    }
}

struct DetailedDisclaimerView: View {
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            ScrollView {
                VStack(alignment: .leading, spacing: 20) {
                    Text("Comprehensive Medical Disclaimer")
                        .font(.title2)
                        .fontWeight(.bold)
                        .padding(.bottom)
                    
                    Group {
                        DetailedSection(
                            title: "App Store Compliance",
                            content: """
                            This application complies with Apple's App Store Review Guidelines regarding medical and health-related content. It does not claim to diagnose, cure, treat, or prevent any medical condition.
                            """
                        )
                        
                        DetailedSection(
                            title: "Statistical Limitations",
                            content: """
                            All calculations are based on actuarial tables and population statistics. These models cannot account for individual health circumstances, genetic factors, lifestyle changes, or medical interventions that may significantly affect individual outcomes.
                            """
                        )
                        
                        DetailedSection(
                            title: "Privacy and Data",
                            content: """
                            Your health data is processed locally on your device. We do not store, transmit, or share personal health information. You maintain full control over your data and can delete it at any time.
                            """
                        )
                        
                        DetailedSection(
                            title: "Emergency Situations",
                            content: """
                            This app is not designed for emergency situations. If you are experiencing a medical emergency, contact emergency services immediately (911 in the US, or your local emergency number).
                            """
                        )
                        
                        DetailedSection(
                            title: "Liability Disclaimer",
                            content: """
                            The developers, publishers, and distributors of this app disclaim all liability for any decisions made based on the app's output. Use of this app is at your own risk.
                            """
                        )
                        
                        DetailedSection(
                            title: "Updates and Changes",
                            content: """
                            This disclaimer may be updated periodically. Continued use of the app after updates constitutes acceptance of the revised disclaimer.
                            """
                        )
                    }
                    
                    Text("Last Updated: \(Date().formatted(date: .abbreviated, time: .omitted))")
                        .font(.caption)
                        .foregroundColor(.secondary)
                        .padding(.top)
                }
                .padding()
            }
            .navigationTitle("Full Disclaimer")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .confirmationAction) {
                    Button("Done") { dismiss() }
                }
            }
        }
    }
}

struct DetailedSection: View {
    let title: String
    let content: String
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            Text(title)
                .font(.headline)
                .fontWeight(.semibold)
            
            Text(content)
                .font(.body)
                .foregroundColor(.secondary)
        }
    }
}

// MARK: - Disclaimer Manager
class DisclaimerManager: ObservableObject {
    @Published var hasAcceptedDisclaimer: Bool = false
    
    init() {
        checkDisclaimerStatus()
    }
    
    func checkDisclaimerStatus() {
        hasAcceptedDisclaimer = UserDefaults.standard.bool(forKey: "hasAcceptedMedicalDisclaimer")
    }
    
    func resetDisclaimer() {
        UserDefaults.standard.removeObject(forKey: "hasAcceptedMedicalDisclaimer")
        UserDefaults.standard.removeObject(forKey: "disclaimerAcceptanceDate")
        UserDefaults.standard.removeObject(forKey: "disclaimerAcceptanceLog")
        hasAcceptedDisclaimer = false
    }
    
    func getAcceptanceDate() -> Date? {
        return UserDefaults.standard.object(forKey: "disclaimerAcceptanceDate") as? Date
    }
    
    var isDisclaimerRequired: Bool {
        // Show disclaimer if not accepted or if it's been more than 90 days
        guard hasAcceptedDisclaimer,
              let acceptanceDate = getAcceptanceDate() else {
            return true
        }
        
        let daysSinceAcceptance = Calendar.current.dateComponents([.day], from: acceptanceDate, to: Date()).day ?? 0
        return daysSinceAcceptance > 90
    }
}

// MARK: - Preview
#Preview {
    DisclaimerView(hasAcceptedDisclaimer: .constant(false))
}
